<!DOCTYPE html>
<html lang="hu">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Programkód Megjelenítése</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            background-color: #f4f4f4;
            padding: 20px;
        }
        .code-container {
            background: #282c34;
            color: #abb2bf;
            padding: 15px;
            border-radius: 5px;
            font-family: "Courier New", monospace;
            overflow-x: auto;
            white-space: pre;
        }
    </style>
</head>
<body>

    <h2>Python Kód Példa</h2>
    <div class="code-container">
        <pre><code>

"""
===============================================================================
    Projekt: Moduláris Esőkert Számítási Programok
    Fejlesztő: Glia Kutatás - Fejlesztési Innovációs Kft.
    Weboldal: www.glia.hu
    Verzió: 1.0
    Dátum: 2018
    Licenc: Nyílt forráskódú szoftver (MIT Licenc)

    Pályázati támogatás:
    Ez a fejlesztés a GINOP-2.1.7-15-2016-02435 pályázat keretén belül 
    valósult meg, amely célja a moduláris beton-műanyag kompozit szikkasztó 
    rendszer fejlesztése volt és a Glia Kft. tarnabodi telephelyén került megvalósításra.

    Leírás:
    Ez a program a moduláris esőkert különböző hidrológiai paramétereinek 
    számítására szolgál. Tartalmazza az alábbi funkciókat:
    - Kapilláris vízemelés számítása
    - Infiltráció és vízbeszivárgás számítása
    - Vízmozgás sebességének becslése
    - Párolgás és evapotranszspiráció számítása
    - Esőkert tározási kapacitás és túlcsordulás vizsgálata

    Licencfeltételek:
    Ez a szoftver az MIT Licenc alatt áll, így szabadon másolható, módosítható 
    és terjeszthető, feltéve hogy a fent megnevezett fejlesztői jogokat 
    és a licencfeltételeket tiszteletben tartják.
===============================================================================
"""

import math
import argparse  # Parancssori argumentumok kezelésére

def young_laplace_capillary_rise(surface_tension, contact_angle, pore_diameter, water_density, gravity=9.81):
    """
    Kapilláris emelkedési magasság számítása a Young-Laplace egyenlet alapján.

    Paraméterek:
        surface_tension (float): Víz felületi feszültsége (N/m)
        contact_angle (float): Kontakt szög (fok)
        pore_diameter (float): Talaj pórusának átmérője (m)
        water_density (float): Víz sűrűsége (kg/m³)
        gravity (float, opcionális): Gravitációs gyorsulás (m/s²), alapértelmezett: 9.81

    Visszatérés:
        float: Kapilláris emelkedési magasság (m)
    """
    capillary_height = (4 * surface_tension * math.cos(math.radians(contact_angle))) / (pore_diameter * water_density * gravity)
    return capillary_height

def kozeny_carman_permeability(particle_size, porosity, viscosity, gravity=9.81):
    """
    Hidraulikus vezetőképesség becslése a Kozeny-Carman egyenlet alapján.

    Paraméterek:
        particle_size (float): Talajszemcsék mérete (m)
        porosity (float): Talaj porozitása (dimenzió nélküli)
        viscosity (float): Víz dinamikus viszkozitása (Pa·s)
        gravity (float, opcionális): Gravitációs gyorsulás (m/s²), alapértelmezett: 9.81

    Visszatérés:
        float: Hidraulikus vezetőképesség (m/s)
    """
    permeability = (gravity / viscosity) * (particle_size ** 2) * (porosity ** 3) / (1 - porosity) ** 2
    return permeability

def capillary_rise_max(surface_tension, contact_angle, porosity, viscosity, water_density, permeability, hc):
    """
    Maximális kapilláris emelkedési magasság számítása.

    Paraméterek:
        surface_tension (float): Víz felületi feszültsége (N/m)
        contact_angle (float): Kontakt szög (fok)
        porosity (float): Talaj porozitása
        viscosity (float): Víz viszkozitása (Pa·s)
        water_density (float): Víz sűrűsége (kg/m³)
        permeability (float): Talaj hidraulikus vezetőképessége (m/s)
        hc (float): Kapilláris emelkedési magasság (m)

    Visszatérés:
        float: Maximális kapilláris emelkedési magasság (m)
    """
    hmax = (2 * surface_tension * porosity * math.cos(math.radians(contact_angle))) / (viscosity * water_density * permeability) + hc
    return hmax

def calculate_modular_rain_garden_capillary_rise(particle_size, porosity, surface_tension, contact_angle, viscosity, water_density, gravity=9.81):
    """
    Moduláris esőkert kapilláris vízemelésének számítása bemeneti paraméterek alapján.

    Paraméterek:
        particle_size (float): Talajszemcsék mérete (m)
        porosity (float): Talaj porozitása
        surface_tension (float): Víz felületi feszültsége (N/m)
        contact_angle (float): Kontakt szög (fok)
        viscosity (float): Víz viszkozitása (Pa·s)
        water_density (float): Víz sűrűsége (kg/m³)
        gravity (float, opcionális): Gravitációs gyorsulás (m/s²), alapértelmezett: 9.81

    Visszatérés:
        dict: Számítási eredmények (kapilláris emelkedés, hidraulikus vezetőképesség, maximális kapilláris emelkedés)
    """
    # Hidraulikus vezetőképesség számítása
    permeability = kozeny_carman_permeability(particle_size, porosity, viscosity, gravity)
    
    # Kapilláris emelkedés számítása (Young-Laplace egyenlet alapján)
    hc = young_laplace_capillary_rise(surface_tension, contact_angle, particle_size, water_density, gravity)
    
    # Maximális kapilláris emelkedés számítása
    hmax = capillary_rise_max(surface_tension, contact_angle, porosity, viscosity, water_density, permeability, hc)
    
    return {
        "Kapilláris emelkedés (Young-Laplace)": hc,
        "Hidraulikus vezetőképesség (Kozeny-Carman)": permeability,
        "Maximális kapilláris emelkedési magasság": hmax
    }

def main():
    """
    A fő programfüggvény, amely parancssori argumentumokat dolgoz fel,
    és elvégzi a kapilláris vízemelés számítását moduláris esőkertekhez.
    """
    parser = argparse.ArgumentParser(description="Moduláris esőkert kapilláris vízemelés számítása.")
    
    # Parancssori argumentumok meghatározása
    parser.add_argument("--surface_tension", type=float, required=True, help="Víz felületi feszültsége (N/m)")
    parser.add_argument("--contact_angle", type=float, required=True, help="Kontakt szög (fokban)")
    parser.add_argument("--particle_size", type=float, required=True, help="Pórus átmérője (m)")
    parser.add_argument("--water_density", type=float, required=True, help="Víz sűrűsége (kg/m³)")
    parser.add_argument("--porosity", type=float, required=True, help="Porozitás (dimenzió nélküli)")
    parser.add_argument("--viscosity", type=float, required=True, help="Dinamikus viszkozitás (Pa·s)")

    # Argumentumok beolvasása
    args = parser.parse_args()

    # Számítások végrehajtása
    results = calculate_modular_rain_garden_capillary_rise(
        args.particle_size, args.porosity, args.surface_tension, 
        args.contact_angle, args.viscosity, args.water_density
    )

    # Eredmények kiírása
    print("\nKapilláris vízemelés számítás eredményei:")
    print(f"Kapilláris emelkedési magasság (Young-Laplace): {results['Kapilláris emelkedés (Young-Laplace)']:.2f} m")
    print(f"Hidraulikus vezetőképesség (Kozeny-Carman): {results['Hidraulikus vezetőképesség (Kozeny-Carman)']:.6e} m/s")
    print(f"Maximális kapilláris emelkedési magasság: {results['Maximális kapilláris emelkedési magasság']:.2f} m")

if __name__ == "__main__":
    main()
  
  
  
  </code></pre>
    </div>

</body>
</html>